#include <iostream>
#include <iomanip>
#include <ctime>
#include <cstdlib> // srand
#include <cstring> // parsing command line (strcmp, strlen)
#include <cstdlib> // atexit

#include <algorithm>
#include <numeric>

#include "term.h"
#include "global.h"
#include "front_end.h"
//#include "propositional.h"
#include "functor_tbl.h"
#include "tokenizer.h"
#include "benchmark.h"
#include "program.h"



void print_help(const char* prog_name);
lp::data quick_tokenize(const string& s);


int main(int argc, char* argv[])
{
	using namespace std;
	using namespace lp;

#ifndef NDEBUG
	atexit(check_memory); // check memory upon exiting the program
#endif
	PATH_TO_PROGRAM = boost::filesystem::canonical(argv[0]);
	CURRENT_PATH = boost::filesystem::current_path(); // initialize CURRENT_PATH
	//std::cerr << "argv[0]: " << argv[0] << "\n";
	//std::cerr << "path: " << CURRENT_PATH << "\n";

	try {
		// const time_t start_time = time(nullptr);
		srand(static_cast<unsigned>(time(nullptr)));

#ifndef NDEBUG
		cerr << "WARNING: compiled without debugging flag NDEBUG, heavy loss of performance\n";
#endif

		// Create default program
		Program kb;

		// Read Default Program
		std::fstream fin("default.pl");
		if (fin.is_open()) {
			try {
				fin >> kb;
			} catch (const parse_error& pe) {
				cerr << "Parse Error while reading \"default.pl\": " << pe.what() << "\n";
				throw destruct_and_exit();
			} catch (...) {
				cerr << "Error: failed to read \"default.pl\"\n";
				throw destruct_and_exit();
			}
		} else {
			cerr << "Warning: could not open file \"default.pl\"\n";
		}
		fin.close();


		// Read commands from terminal
		bool is_interactive = true; // interactive mode?
		int benchmarks = 1; // how many benchmarks to run (0=no benchmarking)

		for (int i = 1; i < argc; ++i) {
			if (argv[i][0] == switch_char) {
				const auto arglen = strlen(argv[i]);
				if (arglen < 2) {
					cerr << "Error: need to supply commands to " << switch_char << "\n";
					exit(1);
				}
				if (argv[i][1] == switch_char) {
					// Long option
					if (arglen == 2) { cerr << "Error: " << switch_char << switch_char << " not a valid command\n"; exit(1); }
					if (i+1 >= argc) { cerr << "Error: argument missing\n"; exit(1); }
					data& dat = kb.params[ &argv[i][2] ]; // Note: this may throw
					dat = quick_tokenize( argv[i+1] ); // Note: this may throw
					++i;
					continue;
				} else {
					// Short option
					for (char* opt = &argv[i][1]; *opt != '\0'; ++opt) {
						switch (*opt) {
						case 'I': is_interactive = true; break;
						case 'N': is_interactive = false; break;
						case 'C': kb.clear(); break; // clear knowledge base
						case 'V': cout << PROG_NAME << " "<< PROG_VERSION << "\n"; exit(0);
						case 'H': print_help(argv[0]); exit(0);
						case 'o':
							if (*(opt+1) != '\0') {
								cerr << "Error: o must be last character in param string\n";
								exit(1);
							}
							if (i+1 < argc) {
								++i;
								ofstream outf(argv[i]);
								if (!outf) { 
									DEBUG_WARNING(cerr << "Error: could not open file " << argv[i] << "\n");
									DEBUG_INFO(cerr << "Knowledge Base:\n" << kb << "\nError opening file\n");
									exit(1);
								}
								outf << kb << "\n";
								if (!outf) {
									DEBUG_WARNING(cerr << "Error: could not properly write to output file " << argv[i] << "\n");
									DEBUG_INFO(cerr << "Dumping knowledge base:\n" << kb << "Error writing to output file\n");
									exit(1); 
								}
								outf.close();
								DEBUG_INFO(cout << "Wrote knowledge base to file \"" << argv[i] << "\"\n");
							} else { cerr << "Error: no output file supplied\n"; exit(1); }
							break;
						case 'B': {
							// syntax: /B 30 nrsample
							if (i+2 >= argc) {
								cerr << "Error: no benchmark size/method supplied\n"; 
								exit(1);
							}
							// Read count
							++i;
							int count;
							try {
								count = destringify<int>(string(argv[i]));
							} catch (conversion_error) {
								cerr << "Error: non-numeric benchmark size\n";
								exit(1);
							}
							// Execute benchmark
							++i;
							kb = benchmark(count,argv[i],kb);
							break;
								  }
						case 'X': {
							// cross validation, syntax: /X method path_to_folder
							if (i+2 >= argc) {
								cerr << "Error: incorrect syntax for cross validation, use /X method path_to_folder\n"; 
								exit(1);
							}
							// Read method
							++i;
							const std::string method = argv[i];
							++i;
							boost::filesystem::path path = argv[i];
							// Execute Cross validation
							validate_rec(method,kb,path);
							break;
								  }
						case 'A': {
							// auto cross validation, syntax: /A method_mask folds repeat
							if (i+3 >= argc) {
								cerr << "Error: incorrect syntax for auto cross validation, use /A method_mask folds repeat\n"; 
								exit(1);
							}
							++i;
							const std::string method_mask = argv[i];
							++i;
							int folds;
							try {
								folds = destringify<int>(string(argv[i]));
							} catch (conversion_error) {
								cerr << "Error: non-numeric fold size\n";
								exit(1);
							}
							++i;
							int repeat;
							try {
								repeat = destringify<int>(string(argv[i]));
							} catch (conversion_error) {
								cerr << "Error: non-numeric value for repeat\n";
								exit(1);
							}
							auto_cross_validation(repeat,folds,method_mask,kb);
							break;
								  }
						case 'G':  {
							if (*(opt+1) != '\0') {
								cerr << "Error: G must be last character in param string\n";
								exit(1);
							}
							if (++i >= argc) {
								cerr << "Error: generalization method not provided\n"; exit(1);
							}
							// Note: this may throw invalid_name
							try {
								kb.generalize(argv[i]);
							} catch (invalid_name) {
								cerr << "Error: unknown generalization method: " << argv[i] << "\n";
								throw destruct_and_exit();
							} catch (induction_exception) {
								cerr << "Error: induction could not complete\n";
								throw destruct_and_exit();
							}
							break;
								   }
						default: {
							if (arglen != 2) { cerr << "Error: unknown command " << argv[i] << "\n"; exit(1); }
							if (i+1 >= argc) { cerr << "Error: need an argument for " << argv[i] << "\n"; exit(1); }
							data& dat = kb.params[*opt]; // may throw
							dat = quick_tokenize(argv[i+1]); // may throw
							++i;
								 } // default block
						} // switch
					} // loop through all letter commmands after switch char
				} // short or long name?
			} else {
				// not switch char, this is an input file
				try {
					if (!consult_file(argv[i],kb)) {
						cerr << "Error: failed to consult file \"" << argv[i] << "\"\n";
						throw destruct_and_exit();
					}
					fin >> kb;
				} catch (const parse_error& pe) {
					cerr << "Parse Error while reading " << argv[i] << ": " << pe.what() << "\n";
					throw destruct_and_exit();
				} catch (destruct_and_exit) {
					throw;
				} catch (...) {
					cerr << "Error: failed to read file \"" << argv[i] << "\"\n";
					throw destruct_and_exit();
				}
			}
		} // loop through command line inputs

		// run front end in interactive mode
		if (is_interactive) {
			Front_End fe(std::move(kb));
			fe.interactive();
		}

		// cout << "Program execution time: " << ( time(nullptr) - start_time ) << " seconds" << endl;
	} catch (destruct_and_exit) {
		DEBUG_TRACE(cerr << "Caught destruct_and_exit\n");
	} catch (const invalid_name& err) {
		cerr << "Error: invalid name: " << err.what() << "\n";
	} catch (const conversion_error& err) {
		cerr << "Error: conversion " << err.what() << " (should NOT be caught here!)\n";
	} catch (const parse_error& err) {
		cerr << "Error: parse_error " << err.what() << " (should NOT be caught here!)\n";
	} catch (const std::exception& err) {
		cerr << "Error: std::exception " << err.what() << " (should NOT be caught here!)\n";
	} catch (...) {
		cerr << "Error: unknown error caught (this is most likely a bug in the program)\n";
	}

	return 0;
}




void print_help(const char* prog_name)
{
	cout << "Welcome to " << PROG_NAME << " version " << PROG_VERSION << ", by John Ahlgren <" << AUTHOR_EMAIL << ">\n";
	cout << "Usage: " << prog_name << " [" << switch_char << "<options>] [<arguments>] infile...\n";
	cout << "Option Description\n";
	cout << "H      Print this help menu and exit\n";
	cout << "V      Print version and exit\n";
	cout << "I      Run in interactive mode\n";
	cout << "N      Non-interactive mode (exit after commands)\n";
	cout << "q      Run quietly (verbosity level 0)\n";
	cout << "g      Generalize knowledge base\n";
	cout << "F      Set purely functional mode (advanced)\n";
	cout << "\n";
	cout << "Option Argument(s) Description\n";
	cout << "v      <int>       Verbosity level\n";
	cout << "p      <int>       Pretty printing level\n";
	cout << "z      <int>       Post-compression level after generalizing\n";
	cout << "o      <filename>  Output knowledge base to file\n";
	cout << "s      <par> <val> Query ?- set(par,val) to set parameter with value\n";
}



lp::data quick_tokenize(const string& s)
{
	using namespace lp;

	stringstream ss;
	ss << s;
	long long line = 1;
	Token tok;
	try {
		std::set<id_type> qvars;
		tok = read_atom(ss,line,qvars,lp::prolog_dic);
	} catch (const parse_error& pe) {
		cerr << "Parse error while reading atom " << s << ": " << pe.what() << "\n";
		throw destruct_and_exit();
	}
	//std::cerr << "Quick tokenize, data: " << tok.get_data() << "\n";
	return tok.get_data();
}

